from flask import Flask, render_template, request, redirect, url_for, session, flash, jsonify
import sqlite3
from datetime import datetime, date
import hashlib
import os

app = Flask(__name__)
app.secret_key = 'kidro_secret_key_2024'

# Database configuration
DATABASE = 'db/kidro.db'

def get_db_connection():
    conn = sqlite3.connect(DATABASE)
    conn.row_factory = sqlite3.Row
    return conn

def init_db():
    """Initialize the database with all required tables"""
    if not os.path.exists('db'):
        os.makedirs('db')
    
    conn = get_db_connection()
    
    # Users table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL,
            role TEXT NOT NULL CHECK(role IN ('admin', 'teacher', 'parent')),
            name TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ''')
    
    # Children table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS children (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            age INTEGER NOT NULL,
            class TEXT NOT NULL,
            parent_id INTEGER,
            health_info TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (parent_id) REFERENCES users (id)
        )
    ''')
    
    # Attendance table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS attendance (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            child_id INTEGER NOT NULL,
            date DATE NOT NULL,
            status TEXT NOT NULL CHECK(status IN ('present', 'absent', 'late')),
            notes TEXT,
            recorded_by INTEGER,
            FOREIGN KEY (child_id) REFERENCES children (id),
            FOREIGN KEY (recorded_by) REFERENCES users (id)
        )
    ''')
    
    # Payments table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS payments (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            child_id INTEGER NOT NULL,
            amount_paid REAL NOT NULL,
            payment_date DATE NOT NULL,
            note TEXT,
            recorded_by INTEGER,
            FOREIGN KEY (child_id) REFERENCES children (id),
            FOREIGN KEY (recorded_by) REFERENCES users (id)
        )
    ''')
    
    # Programs table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS programs (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            title TEXT NOT NULL,
            program_date DATE NOT NULL,
            description TEXT,
            class TEXT,
            created_by INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (created_by) REFERENCES users (id)
        )
    ''')
    
    # Messages table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS messages (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            sender_id INTEGER NOT NULL,
            receiver_id INTEGER NOT NULL,
            subject TEXT NOT NULL,
            content TEXT NOT NULL,
            sent_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            is_read BOOLEAN DEFAULT FALSE,
            FOREIGN KEY (sender_id) REFERENCES users (id),
            FOREIGN KEY (receiver_id) REFERENCES users (id)
        )
    ''')
    
    # Teachers table
    conn.execute('''
        CREATE TABLE IF NOT EXISTS teachers (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER NOT NULL,
            assigned_class TEXT,
            FOREIGN KEY (user_id) REFERENCES users (id)
        )
    ''')
    
    # Insert default admin user
    admin_password = hashlib.md5('admin123'.encode()).hexdigest()
    conn.execute('''
        INSERT OR IGNORE INTO users (username, password, role, name)
        VALUES (?, ?, ?, ?)
    ''', ('admin', admin_password, 'admin', 'مدير النظام'))
    
    conn.commit()
    conn.close()

def hash_password(password):
    return hashlib.md5(password.encode()).hexdigest()

def check_auth(required_role=None):
    if 'user_id' not in session:
        return False
    if required_role and session.get('role') != required_role:
        return False
    return True

# Routes
@app.route('/')
def index():
    return render_template('index.html')

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = hash_password(request.form['password'])
        
        conn = get_db_connection()
        user = conn.execute(
            'SELECT * FROM users WHERE username = ? AND password = ?',
            (username, password)
        ).fetchone()
        conn.close()
        
        if user:
            session['user_id'] = user['id']
            session['username'] = user['username']
            session['role'] = user['role']
            session['name'] = user['name']
            
            if user['role'] == 'admin':
                return redirect(url_for('admin_dashboard'))
            elif user['role'] == 'teacher':
                return redirect(url_for('teacher_dashboard'))
            elif user['role'] == 'parent':
                return redirect(url_for('parent_dashboard'))
        else:
            flash('اسم المستخدم أو كلمة المرور غير صحيحة', 'error')
    
    return render_template('login.html')

@app.route('/logout')
def logout():
    session.clear()
    flash('تم تسجيل الخروج بنجاح', 'success')
    return redirect(url_for('index'))

# Admin Routes
@app.route('/admin')
def admin_dashboard():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Get statistics
    total_children = conn.execute('SELECT COUNT(*) as count FROM children').fetchone()['count']
    total_teachers = conn.execute('SELECT COUNT(*) as count FROM teachers').fetchone()['count']
    total_parents = conn.execute('SELECT COUNT(*) as count FROM users WHERE role = "parent"').fetchone()['count']
    
    # Today's attendance
    today = date.today()
    present_today = conn.execute(
        'SELECT COUNT(*) as count FROM attendance WHERE date = ? AND status = "present"',
        (today,)
    ).fetchone()['count']
    
    conn.close()
    
    stats = {
        'total_children': total_children,
        'total_teachers': total_teachers,
        'total_parents': total_parents,
        'present_today': present_today
    }
    
    return render_template('admin/dashboard.html', stats=stats)

@app.route('/admin/children')
def admin_children():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    children = conn.execute('''
        SELECT c.*, u.name as parent_name 
        FROM children c 
        LEFT JOIN users u ON c.parent_id = u.id
        ORDER BY c.name
    ''').fetchall()
    conn.close()
    
    return render_template('admin/children.html', children=children)

@app.route('/admin/add_child', methods=['GET', 'POST'])
def admin_add_child():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        name = request.form['name']
        age = request.form['age']
        class_name = request.form['class']
        parent_id = request.form.get('parent_id') or None
        health_info = request.form['health_info']
        
        conn = get_db_connection()
        conn.execute('''
            INSERT INTO children (name, age, class, parent_id, health_info)
            VALUES (?, ?, ?, ?, ?)
        ''', (name, age, class_name, parent_id, health_info))
        conn.commit()
        conn.close()
        
        flash('تم إضافة الطفل بنجاح', 'success')
        return redirect(url_for('admin_children'))
    
    # Get parents for dropdown
    conn = get_db_connection()
    parents = conn.execute('SELECT id, name FROM users WHERE role = "parent"').fetchall()
    conn.close()
    
    return render_template('admin/add_child.html', parents=parents)

@app.route('/admin/attendance')
def admin_attendance():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    selected_date = request.args.get('date', date.today().strftime('%Y-%m-%d'))
    
    conn = get_db_connection()
    attendance_records = conn.execute('''
        SELECT c.id, c.name, c.class, a.status, a.notes
        FROM children c
        LEFT JOIN attendance a ON c.id = a.child_id AND a.date = ?
        ORDER BY c.class, c.name
    ''', (selected_date,)).fetchall()
    conn.close()
    
    return render_template('admin/attendance.html', 
                         attendance_records=attendance_records, 
                         selected_date=selected_date)

@app.route('/admin/record_attendance', methods=['POST'])
def admin_record_attendance():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    attendance_date = request.form['date']
    
    conn = get_db_connection()
    
    # Delete existing attendance for this date
    conn.execute('DELETE FROM attendance WHERE date = ?', (attendance_date,))
    
    # Insert new attendance records
    for key, value in request.form.items():
        if key.startswith('status_'):
            child_id = key.split('_')[1]
            status = value
            notes = request.form.get(f'notes_{child_id}', '')
            
            conn.execute('''
                INSERT INTO attendance (child_id, date, status, notes, recorded_by)
                VALUES (?, ?, ?, ?, ?)
            ''', (child_id, attendance_date, status, notes, session['user_id']))
    
    conn.commit()
    conn.close()
    
    flash('تم تسجيل الحضور بنجاح', 'success')
    return redirect(url_for('admin_attendance'))

@app.route('/admin/payments')
def admin_payments():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    payments = conn.execute('''
        SELECT p.*, c.name as child_name
        FROM payments p
        JOIN children c ON p.child_id = c.id
        ORDER BY p.payment_date DESC
    ''').fetchall()
    conn.close()
    
    return render_template('admin/payments.html', payments=payments)

@app.route('/admin/add_payment', methods=['GET', 'POST'])
def admin_add_payment():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        child_id = request.form['child_id']
        amount = request.form['amount']
        payment_date = request.form['payment_date']
        note = request.form['note']
        
        conn = get_db_connection()
        conn.execute('''
            INSERT INTO payments (child_id, amount_paid, payment_date, note, recorded_by)
            VALUES (?, ?, ?, ?, ?)
        ''', (child_id, amount, payment_date, note, session['user_id']))
        conn.commit()
        conn.close()
        
        flash('تم إضافة الدفعة بنجاح', 'success')
        return redirect(url_for('admin_payments'))
    
    conn = get_db_connection()
    children = conn.execute('SELECT id, name FROM children ORDER BY name').fetchall()
    conn.close()
    
    return render_template('admin/add_payment.html', children=children)

@app.route('/admin/programs')
def admin_programs():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    programs = conn.execute('''
        SELECT p.*, u.name as created_by_name
        FROM programs p
        JOIN users u ON p.created_by = u.id
        ORDER BY p.program_date DESC
    ''').fetchall()
    conn.close()
    
    return render_template('admin/programs.html', programs=programs)

@app.route('/admin/add_program', methods=['GET', 'POST'])
def admin_add_program():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        title = request.form['title']
        program_date = request.form['program_date']
        description = request.form['description']
        class_name = request.form['class']
        
        conn = get_db_connection()
        conn.execute('''
            INSERT INTO programs (title, program_date, description, class, created_by)
            VALUES (?, ?, ?, ?, ?)
        ''', (title, program_date, description, class_name, session['user_id']))
        conn.commit()
        conn.close()
        
        flash('تم إضافة البرنامج بنجاح', 'success')
        return redirect(url_for('admin_programs'))
    
    return render_template('admin/add_program.html')

@app.route('/admin/users')
def admin_users():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    users = conn.execute('SELECT * FROM users WHERE role != "admin" ORDER BY role, name').fetchall()
    conn.close()
    
    return render_template('admin/users.html', users=users)

@app.route('/admin/add_user', methods=['GET', 'POST'])
def admin_add_user():
    if not check_auth('admin'):
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        username = request.form['username']
        password = hash_password(request.form['password'])
        role = request.form['role']
        name = request.form['name']
        
        conn = get_db_connection()
        try:
            conn.execute('''
                INSERT INTO users (username, password, role, name)
                VALUES (?, ?, ?, ?)
            ''', (username, password, role, name))
            
            # If it's a teacher, add to teachers table
            if role == 'teacher':
                user_id = conn.lastrowid
                assigned_class = request.form.get('assigned_class', '')
                conn.execute('''
                    INSERT INTO teachers (user_id, assigned_class)
                    VALUES (?, ?)
                ''', (user_id, assigned_class))
            
            conn.commit()
            flash('تم إضافة المستخدم بنجاح', 'success')
            return redirect(url_for('admin_users'))
        except sqlite3.IntegrityError:
            flash('اسم المستخدم موجود بالفعل', 'error')
        finally:
            conn.close()
    
    return render_template('admin/add_user.html')

# Teacher Routes
@app.route('/teacher')
def teacher_dashboard():
    if not check_auth('teacher'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Get teacher's assigned class
    teacher_info = conn.execute('''
        SELECT assigned_class FROM teachers WHERE user_id = ?
    ''', (session['user_id'],)).fetchone()
    
    assigned_class = teacher_info['assigned_class'] if teacher_info else None
    
    # Get children in teacher's class
    if assigned_class:
        children = conn.execute('''
            SELECT * FROM children WHERE class = ? ORDER BY name
        ''', (assigned_class,)).fetchall()
        
        # Today's attendance for this class
        today = date.today()
        present_today = conn.execute('''
            SELECT COUNT(*) as count FROM attendance a
            JOIN children c ON a.child_id = c.id
            WHERE a.date = ? AND a.status = "present" AND c.class = ?
        ''', (today, assigned_class)).fetchone()['count']
    else:
        children = []
        present_today = 0
    
    conn.close()
    
    return render_template('teacher/dashboard.html', 
                         children=children, 
                         assigned_class=assigned_class,
                         present_today=present_today)

@app.route('/teacher/attendance')
def teacher_attendance():
    if not check_auth('teacher'):
        return redirect(url_for('login'))
    
    selected_date = request.args.get('date', date.today().strftime('%Y-%m-%d'))
    
    conn = get_db_connection()
    
    # Get teacher's assigned class
    teacher_info = conn.execute('''
        SELECT assigned_class FROM teachers WHERE user_id = ?
    ''', (session['user_id'],)).fetchone()
    
    assigned_class = teacher_info['assigned_class'] if teacher_info else None
    
    if assigned_class:
        attendance_records = conn.execute('''
            SELECT c.id, c.name, a.status, a.notes
            FROM children c
            LEFT JOIN attendance a ON c.id = a.child_id AND a.date = ?
            WHERE c.class = ?
            ORDER BY c.name
        ''', (selected_date, assigned_class)).fetchall()
    else:
        attendance_records = []
    
    conn.close()
    
    return render_template('teacher/attendance.html', 
                         attendance_records=attendance_records, 
                         selected_date=selected_date,
                         assigned_class=assigned_class)

@app.route('/teacher/record_attendance', methods=['POST'])
def teacher_record_attendance():
    if not check_auth('teacher'):
        return redirect(url_for('login'))
    
    attendance_date = request.form['date']
    
    conn = get_db_connection()
    
    # Get teacher's assigned class
    teacher_info = conn.execute('''
        SELECT assigned_class FROM teachers WHERE user_id = ?
    ''', (session['user_id'],)).fetchone()
    
    assigned_class = teacher_info['assigned_class'] if teacher_info else None
    
    if assigned_class:
        # Delete existing attendance for this date and class
        conn.execute('''
            DELETE FROM attendance WHERE date = ? AND child_id IN (
                SELECT id FROM children WHERE class = ?
            )
        ''', (attendance_date, assigned_class))
        
        # Insert new attendance records
        for key, value in request.form.items():
            if key.startswith('status_'):
                child_id = key.split('_')[1]
                status = value
                notes = request.form.get(f'notes_{child_id}', '')
                
                conn.execute('''
                    INSERT INTO attendance (child_id, date, status, notes, recorded_by)
                    VALUES (?, ?, ?, ?, ?)
                ''', (child_id, attendance_date, status, notes, session['user_id']))
        
        conn.commit()
        flash('تم تسجيل الحضور بنجاح', 'success')
    
    conn.close()
    return redirect(url_for('teacher_attendance'))

@app.route('/teacher/programs')
def teacher_programs():
    if not check_auth('teacher'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Get teacher's assigned class
    teacher_info = conn.execute('''
        SELECT assigned_class FROM teachers WHERE user_id = ?
    ''', (session['user_id'],)).fetchone()
    
    assigned_class = teacher_info['assigned_class'] if teacher_info else None
    
    if assigned_class:
        programs = conn.execute('''
            SELECT * FROM programs 
            WHERE class = ? OR class = 'all'
            ORDER BY program_date DESC
        ''', (assigned_class,)).fetchall()
    else:
        programs = []
    
    conn.close()
    
    return render_template('teacher/programs.html', programs=programs, assigned_class=assigned_class)

# Parent Routes
@app.route('/parent')
def parent_dashboard():
    if not check_auth('parent'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Get parent's children
    children = conn.execute('''
        SELECT * FROM children WHERE parent_id = ? ORDER BY name
    ''', (session['user_id'],)).fetchall()
    
    conn.close()
    
    return render_template('parent/dashboard.html', children=children)

@app.route('/parent/child/<int:child_id>')
def parent_child_details(child_id):
    if not check_auth('parent'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Verify this child belongs to the logged-in parent
    child = conn.execute('''
        SELECT * FROM children WHERE id = ? AND parent_id = ?
    ''', (child_id, session['user_id'])).fetchone()
    
    if not child:
        flash('غير مسموح بالوصول لهذه البيانات', 'error')
        return redirect(url_for('parent_dashboard'))
    
    # Get attendance records
    attendance = conn.execute('''
        SELECT * FROM attendance WHERE child_id = ? ORDER BY date DESC LIMIT 30
    ''', (child_id,)).fetchall()
    
    # Get payment records
    payments = conn.execute('''
        SELECT * FROM payments WHERE child_id = ? ORDER BY payment_date DESC
    ''', (child_id,)).fetchall()
    
    # Get programs for child's class
    programs = conn.execute('''
        SELECT * FROM programs 
        WHERE class = ? OR class = 'all'
        ORDER BY program_date DESC LIMIT 10
    ''', (child['class'],)).fetchall()
    
    conn.close()
    
    return render_template('parent/child_details.html', 
                         child=child, 
                         attendance=attendance, 
                         payments=payments,
                         programs=programs)

@app.route('/parent/messages')
def parent_messages():
    if not check_auth('parent'):
        return redirect(url_for('login'))
    
    conn = get_db_connection()
    
    # Get messages for this parent
    messages = conn.execute('''
        SELECT m.*, u.name as sender_name
        FROM messages m
        JOIN users u ON m.sender_id = u.id
        WHERE m.receiver_id = ?
        ORDER BY m.sent_date DESC
    ''', (session['user_id'],)).fetchall()
    
    conn.close()
    
    return render_template('parent/messages.html', messages=messages)

@app.route('/parent/send_message', methods=['GET', 'POST'])
def parent_send_message():
    if not check_auth('parent'):
        return redirect(url_for('login'))
    
    if request.method == 'POST':
        subject = request.form['subject']
        content = request.form['content']
        
        conn = get_db_connection()
        
        # Send to admin (assuming admin has ID 1)
        admin = conn.execute('SELECT id FROM users WHERE role = "admin" LIMIT 1').fetchone()
        if admin:
            conn.execute('''
                INSERT INTO messages (sender_id, receiver_id, subject, content)
                VALUES (?, ?, ?, ?)
            ''', (session['user_id'], admin['id'], subject, content))
            conn.commit()
            flash('تم إرسال الرسالة بنجاح', 'success')
        
        conn.close()
        return redirect(url_for('parent_messages'))
    
    return render_template('parent/send_message.html')

if __name__ == '__main__':
    init_db()
    app.run(debug=True, host='0.0.0.0', port=5000)
